<?php
session_start();

$jsonFile = 'jadwal.json';

// Inisialisasi file JSON jika belum ada
if (!file_exists($jsonFile)) {
    file_put_contents($jsonFile, json_encode([]));
}

// Fungsi baca jadwal
function readJadwal() {
    global $jsonFile;
    $content = file_get_contents($jsonFile);
    return json_decode($content, true) ?? [];
}

// Fungsi simpan jadwal
function saveJadwal($data) {
    global $jsonFile;
    file_put_contents($jsonFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

// Fungsi cek bentrok jadwal
function checkBentrok($hari, $waktuMulai, $waktuSelesai, $kelas, $excludeNo = null) {
    $jadwal = readJadwal();
    foreach ($jadwal as $item) {
        if ($item['hari'] === $hari && $item['kelas'] === $kelas && $item['no'] != $excludeNo) {
            $mulai = strtotime($item['waktu_mulai']);
            $selesai = strtotime($item['waktu_selesai']);
            $newMulai = strtotime($waktuMulai);
            $newSelesai = strtotime($waktuSelesai);
            
            if (!($newSelesai <= $mulai || $newMulai >= $selesai)) {
                return true;
            }
        }
    }
    return false;
}

// Proses tambah jadwal
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'add') {
        $jadwal = readJadwal();
        
        if (checkBentrok($_POST['hari'], $_POST['waktu_mulai'], $_POST['waktu_selesai'], $_POST['kelas'])) {
            $_SESSION['error'] = 'Jadwal bentrok dengan jadwal yang sudah ada!';
        } else {
            $nos = array_column($jadwal, 'no');
            $noMax = (!empty($nos)) ? max($nos) : 0;
            $jadwal[] = [
                'no' => $noMax + 1,
                'hari' => $_POST['hari'],
                'kelas' => $_POST['kelas'],
                'waktu_mulai' => $_POST['waktu_mulai'],
                'waktu_selesai' => $_POST['waktu_selesai'],
                'mata_pelajaran' => $_POST['mata_pelajaran'],
                'kurikulum' => $_POST['kurikulum'],
                'sekolah' => $_POST['sekolah'],
                'keterangan' => $_POST['keterangan']
            ];
            saveJadwal($jadwal);
            $_SESSION['success'] = 'Jadwal berhasil ditambahkan!';
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    elseif ($action === 'delete' && isset($_POST['no'])) {
        $jadwal = readJadwal();
        $jadwal = array_filter($jadwal, fn($item) => $item['no'] != $_POST['no']);
        saveJadwal(array_values($jadwal));
        $_SESSION['success'] = 'Jadwal berhasil dihapus!';
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
    
    elseif ($action === 'update' && isset($_POST['no'])) {
        $jadwal = readJadwal();
        $key = array_search($_POST['no'], array_column($jadwal, 'no'));
        
        if ($key !== false) {
            if (checkBentrok($_POST['hari'], $_POST['waktu_mulai'], $_POST['waktu_selesai'], $_POST['kelas'], $_POST['no'])) {
                $_SESSION['error'] = 'Jadwal bentrok dengan jadwal yang sudah ada!';
            } else {
                $jadwal[$key] = [
                    'no' => $_POST['no'],
                    'hari' => $_POST['hari'],
                    'kelas' => $_POST['kelas'],
                    'waktu_mulai' => $_POST['waktu_mulai'],
                    'waktu_selesai' => $_POST['waktu_selesai'],
                    'mata_pelajaran' => $_POST['mata_pelajaran'],
                    'kurikulum' => $_POST['kurikulum'],
                    'sekolah' => $_POST['sekolah'],
                    'keterangan' => $_POST['keterangan']
                ];
                saveJadwal($jadwal);
                $_SESSION['success'] = 'Jadwal berhasil diperbarui!';
            }
        }
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

$jadwalList = readJadwal();

// Urutkan jadwal berdasarkan hari dalam seminggu
$urutanHari = ['Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu', 'Minggu'];
usort($jadwalList, function($a, $b) use ($urutanHari) {
    $posisiA = array_search($a['hari'], $urutanHari);
    $posisiB = array_search($b['hari'], $urutanHari);
    
    if ($posisiA !== $posisiB) {
        return $posisiA - $posisiB;
    }
    
    // Jika hari sama, urutkan berdasarkan jam mulai
    return strcmp($a['waktu_mulai'], $b['waktu_mulai']);
});

$editData = null;
if (isset($_GET['edit'])) {
    $editNo = $_GET['edit'];
    $editData = array_values(array_filter($jadwalList, fn($item) => $item['no'] == $editNo))[0] ?? null;
}

$deleteData = null;
if (isset($_GET['delete'])) {
    $deleteNo = $_GET['delete'];
    $deleteData = array_values(array_filter($jadwalList, fn($item) => $item['no'] == $deleteNo))[0] ?? null;
}

$viewData = null;
if (isset($_GET['view'])) {
    $viewNo = $_GET['view'];
    $viewData = array_values(array_filter($jadwalList, fn($item) => $item['no'] == $viewNo))[0] ?? null;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Jadwal Pelajaran</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px 30px;
            border-radius: 15px;
            margin-bottom: 30px;
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3);
        }
        
        .header h1 {
            font-size: 2em;
            margin-bottom: 8px;
            font-weight: 700;
        }
        
        .header p {
            font-size: 1em;
            opacity: 0.95;
        }
        
        .alert {
            padding: 15px 20px;
            margin-bottom: 20px;
            border-radius: 8px;
            font-weight: 500;
            border-left: 5px solid;
            animation: slideDown 0.3s ease;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border-left-color: #28a745;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border-left-color: #dc3545;
        }
        
        .content {
            display: grid;
            grid-template-columns: 1fr;
            gap: 30px;
            margin-bottom: 30px;
        }
        
        .form-section, .table-section {
            background: white;
            padding: 40px;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
        }
        
        .section-header {
            display: flex;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 2px solid #f0f0f0;
        }
        
        .section-header h2 {
            color: #333;
            font-size: 1.5em;
            flex: 1;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        @media (max-width: 768px) {
            .form-row {
                grid-template-columns: 1fr;
            }
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #555;
            font-weight: 600;
            font-size: 0.95em;
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-family: inherit;
            font-size: 0.95em;
            transition: all 0.3s ease;
        }
        
        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: #667eea;
            background: #fafbff;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        textarea {
            resize: vertical;
            min-height: 100px;
        }
        
        .button-group {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        
        button, .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            font-size: 0.95em;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        button:hover, .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-primary:hover {
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .btn-danger {
            background: #dc3545;
            color: white;
        }
        
        .btn-danger:hover {
            background: #c82333;
        }
        
        .btn-info {
            background: #17a2b8;
            color: white;
            padding: 10px 16px;
            font-size: 0.85em;
        }
        
        .btn-info:hover {
            background: #138496;
        }
        
        .btn-eye {
            background: #28a745;
            color: white;
            padding: 10px 16px;
            font-size: 0.85em;
        }
        
        .btn-eye:hover {
            background: #218838;
        }
        
        .table-responsive {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 0;
        }
        
        th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            text-align: left;
            font-size: 0.95em;
            font-weight: 600;
        }
        
        td {
            padding: 15px;
            border-bottom: 1px solid #e0e0e0;
            font-size: 0.95em;
            color: #333;
        }
        
        tr:hover {
            background: #f8f9ff;
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.6);
            animation: fadeIn 0.3s ease;
            overflow-y: auto;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .modal-content {
            background-color: white;
            margin: 3% auto;
            padding: 0;
            border-radius: 15px;
            width: 90%;
            max-width: 600px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
            animation: slideIn 0.3s ease;
        }
        
        @keyframes slideIn {
            from {
                transform: translateY(-50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .modal-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px 30px;
            border-radius: 15px 15px 0 0;
            font-size: 1.3em;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .modal-body {
            padding: 30px;
            max-height: 70vh;
            overflow-y: auto;
        }
        
        .modal-footer {
            display: flex;
            gap: 15px;
            justify-content: flex-end;
            padding: 20px 30px;
            border-top: 1px solid #e0e0e0;
            background: #f8f9fa;
            border-radius: 0 0 15px 15px;
        }
        
        .modal-footer button, .modal-footer .btn {
            padding: 10px 20px;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 600px) {
            .info-grid {
                grid-template-columns: 1fr;
            }
        }
        
        .info-item {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #667eea;
        }
        
        .info-item-label {
            font-size: 0.85em;
            color: #999;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .info-item-value {
            font-size: 1.1em;
            color: #333;
            font-weight: 600;
        }
        
        .info-warning {
            background: #fff3cd;
            border-left-color: #ffc107;
            color: #856404;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .table-stats {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 30px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 1.1em;
            font-weight: 600;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #999;
        }
        
        .empty-state-icon {
            font-size: 4em;
            margin-bottom: 15px;
        }
        
        .empty-state p {
            font-size: 1.1em;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📚 Manajemen Jadwal Pelajaran</h1>
            <p>Kelola jadwal pelajaran dengan mudah, cepat, dan terorganisir</p>
        </div>
        
        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success">
                ✅ <?= $_SESSION['success']; unset($_SESSION['success']); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-error">
                ⚠️ <?= $_SESSION['error']; unset($_SESSION['error']); ?>
            </div>
        <?php endif; ?>
        
        <!-- Modal View Detail -->
        <?php if ($viewData): ?>
            <div id="viewModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        👁️ Detail Jadwal
                    </div>
                    <div class="modal-body">
                        <div class="info-grid">
                            <div class="info-item">
                                <div class="info-item-label">Nomor</div>
                                <div class="info-item-value"><?= $viewData['no']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Hari</div>
                                <div class="info-item-value"><?= $viewData['hari']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Kelas</div>
                                <div class="info-item-value"><?= $viewData['kelas']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Waktu</div>
                                <div class="info-item-value"><?= $viewData['waktu_mulai']; ?> - <?= $viewData['waktu_selesai']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Mata Pelajaran</div>
                                <div class="info-item-value"><?= $viewData['mata_pelajaran']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Kurikulum</div>
                                <div class="info-item-value"><?= $viewData['kurikulum']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Sekolah</div>
                                <div class="info-item-value"><?= $viewData['sekolah']; ?></div>
                            </div>
                        </div>
                        <?php if (!empty($viewData['keterangan'])): ?>
                            <div class="info-item">
                                <div class="info-item-label">Keterangan</div>
                                <div class="info-item-value"><?= nl2br(htmlspecialchars($viewData['keterangan'])); ?></div>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="modal-footer">
                        <a href="<?= $_SERVER['PHP_SELF']; ?>" class="btn btn-secondary">Tutup</a>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Modal Delete -->
        <?php if ($deleteData): ?>
            <div id="deleteModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        ⚠️ Konfirmasi Penghapusan
                    </div>
                    <div class="modal-body">
                        <p style="margin-bottom: 20px; font-size: 1.05em; color: #333;"><strong>Anda yakin ingin menghapus jadwal berikut?</strong></p>
                        <div class="info-grid">
                            <div class="info-item">
                                <div class="info-item-label">No</div>
                                <div class="info-item-value"><?= $deleteData['no']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Hari</div>
                                <div class="info-item-value"><?= $deleteData['hari']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Kelas</div>
                                <div class="info-item-value"><?= $deleteData['kelas']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Waktu</div>
                                <div class="info-item-value"><?= $deleteData['waktu_mulai']; ?> - <?= $deleteData['waktu_selesai']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Mata Pelajaran</div>
                                <div class="info-item-value"><?= $deleteData['mata_pelajaran']; ?></div>
                            </div>
                            <div class="info-item">
                                <div class="info-item-label">Sekolah</div>
                                <div class="info-item-value"><?= $deleteData['sekolah']; ?></div>
                            </div>
                        </div>
                        <div class="info-warning">
                            <strong>⛔ Perhatian!</strong> Aksi ini tidak dapat dibatalkan. Data akan dihapus secara permanen.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <a href="<?= $_SERVER['PHP_SELF']; ?>" class="btn btn-secondary">Batal</a>
                        <form method="POST" style="display: inline;">
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="no" value="<?= $deleteData['no']; ?>">
                            <button type="submit" class="btn btn-danger">Hapus Sekarang</button>
                        </form>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="content">
            <!-- Form Section -->
            <div class="form-section">
                <div class="section-header">
                    <h2><?= $editData ? '✏️ Edit Jadwal' : '➕ Tambah Jadwal Baru' ?></h2>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="<?= $editData ? 'update' : 'add' ?>">
                    <?php if ($editData): ?>
                        <input type="hidden" name="no" value="<?= $editData['no']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="hari">📅 Hari *</label>
                            <select name="hari" id="hari" required>
                                <option value="">-- Pilih Hari --</option>
                                <option value="Senin" <?= ($editData && $editData['hari'] === 'Senin') ? 'selected' : '' ?>>Senin</option>
                                <option value="Selasa" <?= ($editData && $editData['hari'] === 'Selasa') ? 'selected' : '' ?>>Selasa</option>
                                <option value="Rabu" <?= ($editData && $editData['hari'] === 'Rabu') ? 'selected' : '' ?>>Rabu</option>
                                <option value="Kamis" <?= ($editData && $editData['hari'] === 'Kamis') ? 'selected' : '' ?>>Kamis</option>
                                <option value="Jumat" <?= ($editData && $editData['hari'] === 'Jumat') ? 'selected' : '' ?>>Jumat</option>
                                <option value="Sabtu" <?= ($editData && $editData['hari'] === 'Sabtu') ? 'selected' : '' ?>>Sabtu</option>
                                <option value="Minggu" <?= ($editData && $editData['hari'] === 'Minggu') ? 'selected' : '' ?>>Minggu</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="kelas">🏫 Kelas *</label>
                            <input type="text" name="kelas" id="kelas" placeholder="Contoh: X A" value="<?= $editData['kelas'] ?? '' ?>" required>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="waktu_mulai">🕐 Waktu Mulai *</label>
                            <input type="time" name="waktu_mulai" id="waktu_mulai" value="<?= $editData['waktu_mulai'] ?? '' ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="waktu_selesai">🕑 Waktu Selesai *</label>
                            <input type="time" name="waktu_selesai" id="waktu_selesai" value="<?= $editData['waktu_selesai'] ?? '' ?>" required>
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label for="mata_pelajaran">📖 Mata Pelajaran *</label>
                            <input type="text" name="mata_pelajaran" id="mata_pelajaran" placeholder="Contoh: Matematika" value="<?= $editData['mata_pelajaran'] ?? '' ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="kurikulum">📋 Kurikulum *</label>
                            <select name="kurikulum" id="kurikulum" required>
                                <option value="">-- Pilih Kurikulum --</option>
                                <option value="K13" <?= ($editData && $editData['kurikulum'] === 'K13') ? 'selected' : '' ?>>Kurikulum 2013</option>
                                <option value="Merdeka" <?= ($editData && $editData['kurikulum'] === 'Merdeka') ? 'selected' : '' ?>>Kurikulum Merdeka</option>
                                <option value="Internasional" <?= ($editData && $editData['kurikulum'] === 'Internasional') ? 'selected' : '' ?>>Internasional</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="sekolah">🏢 Nama Sekolah *</label>
                        <input type="text" name="sekolah" id="sekolah" placeholder="Nama Sekolah" value="<?= $editData['sekolah'] ?? '' ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="keterangan">💬 Keterangan</label>
                        <textarea name="keterangan" id="keterangan" placeholder="Catatan tambahan, guru pengajar, ruang kelas, dll..."><?= $editData['keterangan'] ?? '' ?></textarea>
                    </div>
                    
                    <div class="button-group">
                        <button type="submit" class="btn btn-primary">
                            <?= $editData ? '💾 Perbarui Jadwal' : '➕ Tambah Jadwal' ?>
                        </button>
                        <?php if ($editData): ?>
                            <a href="<?= $_SERVER['PHP_SELF']; ?>" class="btn btn-secondary">❌ Batal</a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Table Section -->
            <div class="table-section">
                <div class="section-header">
                    <h2>📋 Daftar Jadwal</h2>
                </div>
                
                <?php if (!empty($jadwalList)): ?>
                    <div class="table-stats">
                        📊 Total Jadwal: <strong><?= count($jadwalList); ?></strong>
                    </div>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>No</th>
                                    <th>Hari</th>
                                    <th>Kelas</th>
                                    <th>Waktu</th>
                                    <th>Mata Pelajaran</th>
                                    <th>Sekolah</th>
                                    <th>Aksi</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($jadwalList as $item): ?>
                                    <tr>
                                        <td><strong><?= $item['no']; ?></strong></td>
                                        <td><?= $item['hari']; ?></td>
                                        <td><?= $item['kelas']; ?></td>
                                        <td><?= $item['waktu_mulai']; ?> - <?= $item['waktu_selesai']; ?></td>
                                        <td><?= $item['mata_pelajaran']; ?></td>
                                        <td><?= $item['sekolah']; ?></td>
                                        <td>
                                            <div class="action-buttons">
                                                <a href="?view=<?= $item['no']; ?>" class="btn btn-eye" title="Lihat Detail">👁️ Lihat</a>
                                                <a href="?edit=<?= $item['no']; ?>" class="btn btn-info" title="Edit">✏️ Edit</a>
                                                <a href="?delete=<?= $item['no']; ?>" class="btn btn-danger" title="Hapus">🗑️ Hapus</a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">📭</div>
                        <p><strong>Belum ada jadwal</strong></p>
                        <p style="color: #ccc; font-size: 0.95em;">Silakan tambahkan jadwal baru menggunakan form di atas</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>